using System;
using System.IO;
using System.Xml;
using System.Collections;
using System.Diagnostics;

using Team_Project.Elements;
using Team_Project.PersistencyManagers.Storages;

namespace Team_Project.PersistencyManagers.Protocols
{
	/// <summary>
	/// Protocollo per la lettura e scrittura locale. Il protocollo  ottimizzato
	/// per l'accesso agli elementi quando no vi sia necessit di replicazione.
	/// </summary>
	public class LocalRWProtocol : MarshalByRefObject, IPersistencyProtocol
	{
		/// <summary>
		/// Sotrage locale di default.
		/// </summary>
		protected IStorage storage;
		/// <summary>
		/// Progetto per il quale il protocollo  stato istanziato
		/// </summary>
		protected string project;
		/// <summary>
		/// Tipo di elemento di default per il quale il protocollo  stao istanziato
		/// </summary>
		protected Type elemType;

		/// <summary>
		/// Costruisce un'istanza del protocollo locale.
		/// </summary>
		/// <param name="proj">Nome del progetto che verr acceduto attraverso questa
		/// istanza di protocollo</param>
		/// <param name="elt">Tipo dell'elemento per il quale il protocollo
		/// sta per essere istanziato</param>
		/// <param name="cfg">Nodo xml contenente i dati per la configurazione</param>
		public LocalRWProtocol(string proj,Type elt, XmlNode cfg)
		{
			Trace.WriteLine(Name + " created for prj: " + proj);
			project = proj;
			elemType = elt;
		}

		#region IPersistencyProtocol Members
		/// <summary>
		/// Restituisce "LocalRWProtocol"
		/// </summary>
		public string Name
		{
			get
			{
				return "LocalRWProtocol";
			}
		}
		/// <summary>
		/// Inizializza lo storage di default
		/// </summary>
		public void Intit()
		{
			storage = Globals.Instance.Storages.GetStorageFor(project,elemType);
		}

		/// <summary>
		/// Rilascia lo storage.
		/// </summary>
		public void Close()
		{
			storage = null;
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.Store">
		/// IPersistencyProtocol.Store</see>
		/// </summary>
		/// <param name="el"></param>
		public void Store(IProjectElement el)
		{
			Globals.Instance.Mutexes.RequestWrite(project,el.Location,el.Name);
			try{
				el.WriteTo(storage);}
			finally{
				Globals.Instance.Mutexes.ReleaseWrite(project,el.Location,el.Name);}
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.Retrive">
		/// IPersistencyProtocol.Retrive</see>
		/// </summary>
		/// <param name="location"></param>
		/// <param name="name"></param>
		/// <param name="elementType"></param>
		/// <returns></returns>
		public IProjectElement Retrive(string location,string name, Type elementType)
		{
			IProjectElement el = (IProjectElement)Activator.CreateInstance(elementType,new object[] {project,location,name});
			Globals.Instance.Mutexes.RequestRead(project,location,name);
			try{
				el.LoadFrom(storage);}
			finally{Globals.Instance.Mutexes.ReleaseRead(project,location,name);}
			return el;
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.QueryLocationExists">
		/// IPersistencyProtocol.QueryLocationExists</see>
		/// </summary>
		/// <param name="location"></param>
		/// <returns></returns>
		public bool QueryLocationExists(string location)
		{
			return storage.LocationExists(location);
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.QueryElementExists">
		/// IPersistencyProtocol.QueryElementExists</see>
		/// </summary>
		/// <param name="location"></param>
		/// <param name="name"></param>
		/// <returns></returns>
		public bool QueryElementExists(string location, string name)
		{
			IProjectElement p = (IProjectElement)Activator.CreateInstance(elemType,new object[]{project,location,name});
			return storage.Exists(location,name+p.Suffix);
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.CreateLocation">
		/// IPersistencyProtocol.CreateLocation</see>
		/// </summary>
		/// <param name="location"></param>
		public void CreateLocation(string location)
		{
			storage.CreateLocation(location);
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.DestroyLocation">
		/// IPersistencyProtocol.DestroyLocation</see>
		/// </summary>
		/// <param name="location"></param>
		/// <param name="checkEmpty"></param>
		public void DestroyLocation(string location, bool checkEmpty)
		{
			storage.DestroyLocation(location,checkEmpty);
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.DestroyElement">
		/// IPersistencyProtocol.DestroyElement</see>
		/// </summary>
		/// <param name="el"></param>
		public void DestroyElement(IProjectElement el)
		{
			Globals.Instance.Mutexes.RequestWrite(project,el.Location,el.Name);
			try{
				el.DestroyFrom(storage);}
			finally{Globals.Instance.Mutexes.ReleaseWrite(project,el.Location,el.Name);}
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.QueryElemensInLocation">
		/// IPersistencyProtocol.QueryElemensInLocation</see>
		/// </summary>
		/// <param name="location"></param>
		/// <returns></returns>
		public string[] QueryElemensInLocation(string location)
		{
			string [] res = storage.ElementsIn(location);
			ArrayList al = new ArrayList(res.Length);
			IProjectElement el = (IProjectElement)Activator.CreateInstance(elemType,new object[] {project,location,"TEMP"});
			foreach(string s in res)
			{
				if(s.EndsWith(el.Suffix))
				{
					al.Add(s.Substring(0,s.Length-el.Suffix.Length));
				}
			}
			res = (string []) al.ToArray(typeof(string));
			return res;
		}

		/// <summary>
		/// <see cref="Team_Project.PersistencyManagers.Protocols.IPersistencyProtocol.QueryLocations">
		/// IPersistencyProtocol.QueryLocations</see>
		/// </summary>
		/// <param name="baseL"></param>
		/// <returns></returns>
		public string[] QueryLocations(string baseL)
		{
			return storage.LocationsList(baseL);
		}

		#endregion
	}
}
